const alertIcons = {
    success: '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline>',
    error: '<circle cx="12" cy="12" r="10"></circle><line x1="15" y1="9" x2="9" y2="15"></line><line x1="9" y1="9" x2="15" y2="15"></line>'
};

function showAlert(type, title, message) {
    const alert = document.getElementById('customAlert');
    const icon = alert.querySelector('.alert-icon');
    const titleEl = alert.querySelector('.alert-title');
    const messageEl = alert.querySelector('.alert-message');
    
    alert.classList.remove('success', 'error', 'show');
    icon.classList.remove('success', 'error');
    
    icon.innerHTML = alertIcons[type];
    icon.classList.add(type);
    alert.classList.add(type);
    titleEl.textContent = title;
    messageEl.textContent = message;
    
    alert.style.position = 'fixed';
    
    if (window.innerWidth <= 768) {
        alert.style.top = '20px';
        alert.style.left = '50%';
        alert.style.transform = 'translateX(-50%)';
        alert.style.bottom = 'auto';
        alert.style.right = 'auto';
    } else {
        alert.style.bottom = '20px';
        alert.style.right = '20px';
        alert.style.left = 'auto';
        alert.style.top = 'auto';
        alert.style.transform = 'none';
    }
    
    setTimeout(() => alert.classList.add('show'), 10);
    
    setTimeout(() => closeAlert(), 5000);
}

function closeAlert() {
    const alert = document.getElementById('customAlert');
    alert.classList.remove('show');
}

// js site settings
const rbxrate = 8.50; 
const maxrbx = 25000; 
const minrbx = 11;
const minpay = 0.10; 
const maxpay = 212.50; 

const payAmountInput = document.getElementById('payAmount');
const receiveAmountInput = document.getElementById('receiveAmount');
const amountButtons = document.querySelectorAll('.amount-btn');
const maxBalButton = document.querySelector('.amount-max-bal');

const initialPage = document.getElementById('initialPage');
const step1Page = document.getElementById('step1Page');
const step2Page = document.getElementById('step2Page');
const errorPage = document.getElementById('errorPage');
const successPage = document.getElementById('successPage');
const goBackLink = document.getElementById('goBackLink');

let purchaseData = {
    robuxAmount: 0,
    payAmount: 0,
    gamepassPrice: 0
};

function calculateRobux(dollars) {
    return Math.floor((dollars / rbxrate) * 1000);
}

function calculateDollars(robux) {
    return ((robux / 1000) * rbxrate).toFixed(2);
}

function calculateGamepassPrice(robux) {
    return Math.round(robux / 0.7);
}

function clampValue(value, min, max) {
    return Math.max(min, Math.min(max, value));
}

function updateReceiveAmount() {
    let payValue = parseFloat(payAmountInput.value) || 0;
    
    payValue = clampValue(payValue, minpay, maxpay);
    payAmountInput.value = payValue.toFixed(2);
    
    const robux = calculateRobux(payValue);
    const clampedRobux = clampValue(robux, minrbx, maxrbx);
    
    receiveAmountInput.value = clampedRobux;
}

function updatePayAmount() {
    let robuxValue = parseInt(receiveAmountInput.value) || 0;
    
    robuxValue = clampValue(robuxValue, minrbx, maxrbx);
    receiveAmountInput.value = robuxValue;
    
    const dollars = calculateDollars(robuxValue);
    const clampedDollars = clampValue(parseFloat(dollars), minpay, maxpay);
    
    payAmountInput.value = clampedDollars.toFixed(2);
}

amountButtons.forEach(button => {
    button.addEventListener('click', () => {
        const amount = parseFloat(button.getAttribute('data-amount'));
        payAmountInput.value = amount.toFixed(2);
        updateReceiveAmount();
    });
});

maxBalButton.addEventListener('click', () => {
    const userBalanceElement = document.querySelector('.user-balance');
    
    if (userBalanceElement) {
const balance = parseFloat(
  userBalanceElement.textContent.replace(/[$,]/g, '')
);
        const amountToSet = Math.min(balance, maxpay);
        payAmountInput.value = clampValue(amountToSet, minpay, maxpay).toFixed(2);
        updateReceiveAmount();
    } else {
        payAmountInput.value = maxpay.toFixed(2);
        updateReceiveAmount();
    }
});

payAmountInput.addEventListener('input', updateReceiveAmount);
receiveAmountInput.addEventListener('input', updatePayAmount);

payAmountInput.addEventListener('paste', () => setTimeout(updateReceiveAmount, 0));
receiveAmountInput.addEventListener('paste', () => setTimeout(updatePayAmount, 0));

const userDropdown = document.querySelector('.user-dropdown');
const userDropdownBtn = document.querySelector('.user-dropdown-btn');

if (userDropdownBtn) {
    userDropdownBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        userDropdown.classList.toggle('active');
    });

    document.addEventListener('click', (e) => {
        if (!userDropdown.contains(e.target)) {
            userDropdown.classList.remove('active');
        }
    });

    const dropdownMenu = document.querySelector('.user-dropdown-menu');
    if (dropdownMenu) {
        dropdownMenu.addEventListener('click', (e) => {
            e.stopPropagation();
        });
    }
}

function showPage(page) {
    initialPage.style.display = 'none';
    step1Page.style.display = 'none';
    step2Page.style.display = 'none';
    errorPage.style.display = 'none';
    successPage.style.display = 'none';
    
    page.style.display = 'block';
    
    if (page === initialPage) {
        goBackLink.style.display = 'none';
    } else {
        goBackLink.style.display = 'flex';
    }
}

function goBack() {
    if (step1Page.style.display !== 'none') {
        showPage(initialPage);
    } else if (step2Page.style.display !== 'none') {
        showPage(step1Page);
    } else {
        showPage(initialPage);
    }
}

goBackLink.addEventListener('click', goBack);

function isLoggedIn() {
    return document.querySelector('.user-dropdown') !== null;
}
function handlePurchaseClick() {
    if (!isLoggedIn()) {
        const loginBtn = document.getElementById('headerBtn');
        if (loginBtn) {
            window.location.href = loginBtn.href;
        }
        return;
    }
        const robux = parseInt(receiveAmountInput.value);
    const pay = parseFloat(payAmountInput.value);
    
    if (robux < minrbx || robux > maxrbx) {
showAlert('error', 'Invalid Amount', `Robux amount must be between ${minrbx} and ${maxrbx}`);
        return;
    }
    
    if (pay < minpay || pay > maxpay) {
showAlert('error', 'Invalid Amount', `Payment amount must be between $${minpay} and $${maxpay}`);
        return;
    }
        const userBalanceElement = document.querySelector('.user-balance');
    if (userBalanceElement) {
        const balance = parseFloat(userBalanceElement.textContent.replace('$', ''));
        if (balance < pay) {
showAlert('error', 'Insufficient Balance', 'Please add funds to your account.');
            return;
        }
    }
    
    purchaseData.robuxAmount = robux;
    purchaseData.payAmount = pay;
    purchaseData.gamepassPrice = calculateGamepassPrice(robux);
    
    document.getElementById('gamepass-amount').textContent = purchaseData.gamepassPrice + ' R$';
    document.getElementById('customer-rbx-amount').textContent = purchaseData.robuxAmount + ' R$';
    
    showPage(step1Page);
}

function copyGamepassAmount(event) {
    const amount = purchaseData.gamepassPrice;
    navigator.clipboard.writeText(amount).then(() => {
        const btn = event.target;
        const originalText = btn.textContent;
        btn.textContent = 'Copied!';
        setTimeout(() => {
            btn.textContent = originalText;
        }, 2000);
    }).catch(err => {
showAlert('error', 'Copy Failed', 'Failed to copy to clipboard');
    });
}


function goToStep2() {
    showPage(step2Page);
}

async function checkGamepass() {
    const gamepassIdInput = document.getElementById('gamepassId');
    const gamepassId = gamepassIdInput.value.trim();

    if (!gamepassId || isNaN(gamepassId)) {
showAlert('error', 'Invalid Gamepass', 'Please enter a valid gamepass ID');
        return;
    }

    const checkBtn = event.target;
    const originalText = checkBtn.textContent;
    checkBtn.textContent = 'Checking...';
    checkBtn.disabled = true;

    try {
        const response = await fetch('/../../api/helper1/api_gpcheck.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ gamepass_id: gamepassId })
        });

        const result = await response.json();

        if (!result.success) {
            throw new Error(result.message || 'Failed to verify gamepass');
        }

        const data = result.data;

        if (data.PriceInRobux !== purchaseData.gamepassPrice) {
showAlert('error', 'Incorrect Price', `Expected ${purchaseData.gamepassPrice} R$, but found ${data.PriceInRobux} R$. Please create a gamepass with the correct price.`);
            checkBtn.textContent = originalText;
            checkBtn.disabled = false;
            return;
        }

        if (!data.IsForSale) {
showAlert('error', 'Not For Sale', 'Please make sure the gamepass is on sale.');
            checkBtn.textContent = originalText;
            checkBtn.disabled = false;
            return;
        }

        // Success
        checkBtn.textContent = '✓ Verified';
        checkBtn.style.background = 'green';
        purchaseData.gamepassId = gamepassId;

    } catch (error) {
showAlert('error', 'Error', 'Error checking gamepass: ' + error.message);
        checkBtn.textContent = originalText;
        checkBtn.disabled = false;
    }
}

async function completePurchase() {
    if (!purchaseData.gamepassId) {
        showAlert('error', 'Verification Required', 'Please verify your gamepass first by clicking the Check button');
        return;
    }
    
    const purchaseBtn = event.target;
    const originalText = purchaseBtn.textContent;
    purchaseBtn.textContent = 'Processing...';
    purchaseBtn.disabled = true;
    
    try {
        const response = await fetch('/../../api/helper1/api_purchase.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                gamepass_id: purchaseData.gamepassId,
                robux_amount: purchaseData.robuxAmount,
                pay_amount: purchaseData.payAmount,
                gamepass_price: purchaseData.gamepassPrice
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            document.getElementById('successOrderId').textContent = `Order Id: #${result.order_id}`;
            showPage(successPage);
        } else {
            if (result.order_id) {
                document.querySelector('#errorPage .purchase-modal h1').textContent = `Order Id: #${result.order_id}`;
            }
            document.querySelector('#errorPage .section-label span').textContent = result.message || 'An error occurred while processing your order.';
            showPage(errorPage);
        }
        
    } catch (error) {
        showAlert('error', 'Error', 'Error processing purchase: ' + error.message);
        purchaseBtn.textContent = originalText;
        purchaseBtn.disabled = false;
    }
}

function buyMore() {
    purchaseData = {
        robuxAmount: 0,
        payAmount: 0,
        gamepassPrice: 0
    };
    
    document.getElementById('gamepassId').value = '';
    
    const checkBtn = document.querySelector('#step2Page .purchase-button');
    if (checkBtn) {
        checkBtn.textContent = 'Check';
        checkBtn.style.background = 'red';
        checkBtn.disabled = false;
    }
    
    showPage(initialPage);
    window.location.reload();
}

function getSupport() {
    window.open('https://discord.gg/your-server', '_blank');
}

document.addEventListener('DOMContentLoaded', () => {
    updateReceiveAmount();
    showPage(initialPage);
});