<?php
// IPN Callback handler for NOWPayments
// This file receives payment status updates from NOWPayments

require __DIR__ . '/../../db/config.php';


// Log all incoming requests for debugging
$logFile = 'ipn_log.txt';
$rawInput = file_get_contents('php://input');
file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Raw IPN: " . $rawInput . "\n", FILE_APPEND);

// Get the JSON payload
$ipnData = json_decode($rawInput, true);

// Verify the IPN signature (IMPORTANT for security)
function verifyIPNSignature($ipnData) {
    global $nowpayments_ipn_secret;
    
    // NOWPayments sends a signature in the header
    $receivedSignature = $_SERVER['HTTP_X_NOWPAYMENTS_SIG'] ?? '';
    
    if (empty($receivedSignature)) {
        return false;
    }
    
    // Sort data alphabetically
    ksort($ipnData);
    $sortedData = json_encode($ipnData, JSON_UNESCAPED_SLASHES);
    
    // Calculate expected signature
    $expectedSignature = hash_hmac('sha512', $sortedData, $nowpayments_ipn_secret);
    
    return hash_equals($expectedSignature, $receivedSignature);
}

// Verify signature if IPN secret is configured
if (isset($nowpayments_ipn_secret) && !empty($nowpayments_ipn_secret)) {
    if (!verifyIPNSignature($ipnData)) {
        http_response_code(403);
        file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Invalid signature\n", FILE_APPEND);
        exit('Invalid signature');
    }
}

// Process the IPN
if ($ipnData && isset($ipnData['payment_status'])) {
    try {
        $paymentStatus = $ipnData['payment_status'];
        $orderId = $ipnData['order_id'] ?? null;
        $paymentId = $ipnData['payment_id'] ?? null;
        $actuallyPaid = $ipnData['actually_paid'] ?? null;
        $payCurrency = $ipnData['pay_currency'] ?? null;
        
        file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Processing payment - Status: $paymentStatus, Order: $orderId\n", FILE_APPEND);
        
        if (!$orderId) {
            throw new Exception("No order_id in IPN data");
        }
        
        // Find the deposit by payment address or create new record
        $payAddress = $ipnData['pay_address'] ?? '';
        $stmt = $pdo->prepare("SELECT * FROM deposit WHERE address = ? LIMIT 1");
        $stmt->execute([$payAddress]);
        $deposit = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$deposit) {
            // If not found by address, try to find by order_id in recent deposits
            file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Deposit not found by address, searching recent deposits\n", FILE_APPEND);
            $stmt = $pdo->prepare("SELECT * FROM deposit WHERE created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR) ORDER BY deposit_id DESC LIMIT 10");
            $stmt->execute();
            $recentDeposits = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($recentDeposits as $recentDep) {
                if ($recentDep['address'] === $payAddress) {
                    $deposit = $recentDep;
                    break;
                }
            }
        }
        
        if (!$deposit) {
            throw new Exception("Deposit not found for address: $payAddress");
        }
        
        $depositId = $deposit['deposit_id'];
        $userId = $deposit['depo_owner'];
        
        // Map NOWPayments status to our status
        $newStatus = 'pending';
        $shouldCredit = false;
        
        switch ($paymentStatus) {
            case 'finished':
            case 'confirmed':
                $newStatus = 'completed';
                $shouldCredit = true;
                break;
            case 'failed':
            case 'expired':
            case 'refunded':
                $newStatus = 'canceled';
                break;
            case 'waiting':
            case 'confirming':
            case 'sending':
            case 'partially_paid':
                $newStatus = 'pending';
                break;
            default:
                $newStatus = 'pending';
        }
        
        // Update deposit status
        if ($newStatus === 'completed' || $newStatus === 'canceled') {
            $stmt = $pdo->prepare("UPDATE deposit SET depo_status = ?, finished_at = NOW() WHERE deposit_id = ?");
            $stmt->execute([$newStatus, $depositId]);
        } else {
            $stmt = $pdo->prepare("UPDATE deposit SET depo_status = ? WHERE deposit_id = ?");
            $stmt->execute([$newStatus, $depositId]);
        }
        
        file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Updated deposit $depositId to status: $newStatus\n", FILE_APPEND);
        
        // Credit user balance if payment is completed
        if ($shouldCredit && $deposit['depo_status'] !== 'completed') {
            // Get the USD amount that should be credited
            $priceAmount = $ipnData['price_amount'] ?? $deposit['amount'];
            
            // Update user balance
            $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE discord_id = ?");
            $stmt->execute([$priceAmount, $userId]);
            
            file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Credited $priceAmount USD to user $userId\n", FILE_APPEND);
            
            // Optional: Send notification to Discord bot or user
            // You can add webhook notification here
        }
        
        http_response_code(200);
        echo json_encode(['status' => 'success']);
        
    } catch (Exception $e) {
        file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Error: " . $e->getMessage() . "\n", FILE_APPEND);
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
} else {
    file_put_contents($logFile, date('[Y-m-d H:i:s] ') . "Invalid IPN data\n", FILE_APPEND);
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid IPN data']);
}
?>