<?php
require __DIR__ . '/../../db/config.php';
header('Content-Type: application/json');
    
if (!isset($_SESSION['user']['discord_id'])) {
    header('Location: /../../pages/home.php');
    exit();
}

// Get POST data
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['amount']) || !isset($input['currency'])) {
    echo json_encode(['success' => false, 'error' => 'Invalid request data']);
    exit();
}

$amount = floatval($input['amount']);
$currency = strtolower(trim($input['currency']));
$network = isset($input['network']) ? strtoupper(trim($input['network'])) : null;
$discord_id = $_SESSION['user']['discord_id'];

// Add debug logging
error_log("Discord ID from session: " . var_export($discord_id, true));
error_log("Currency: $currency, Network: " . var_export($network, true));

// Validate discord_id is not empty
if (empty($discord_id)) {
    echo json_encode(['success' => false, 'error' => 'User session invalid - Discord ID not found']);
    exit();
}

// Validate input
if ($amount <= 0) {
    echo json_encode(['success' => false, 'error' => 'Invalid amount']);
    exit();
}

if (empty($currency)) {
    echo json_encode(['success' => false, 'error' => 'Invalid currency']);
    exit();
}

// Convert currency + network to NOWPayments currency code
function getCurrencyCode($currency, $network) {
    $currency = strtolower($currency);
    
    // NOWPayments network-specific currency codes
    // Verified from NOWPayments documentation
    $networkMap = [
        'usdt' => [
            'ERC20' => 'usdterc20',
            'TRC20' => 'usdttrc20',
            'BSC' => 'usdtbsc',
            'BEP20' => 'usdtbsc',       // BEP20 = BSC
            'POLYGON' => 'usdtmatic',
            'MATIC' => 'usdtmatic',
            'SOL' => 'usdtsol',
            'SOLANA' => 'usdtsol'
        ],
        'usdc' => [
            // USDC on NOWPayments:
            // - Base 'usdc' is for ERC20
            // - Other networks have specific codes
            'ERC20' => 'usdc',
            'POLYGON' => 'usdcmatic',
            'MATIC' => 'usdcmatic',
            'SOL' => 'usdcsol',
            'SOLANA' => 'usdcsol',
            'TRC20' => 'usdc',  // Fallback to main USDC
            'BSC' => 'usdc',    // Fallback to main USDC
            'BEP20' => 'usdc'   // Fallback to main USDC
        ],
        'dai' => [
            'ERC20' => 'dai',
            'BSC' => 'dai',
            'BEP20' => 'dai'
        ],
        'busd' => [
            'BEP20' => 'busd',
            'BSC' => 'busd',
            'ERC20' => 'busd'
        ]
    ];
    
    // If currency has network-specific codes and network is provided
    if (isset($networkMap[$currency]) && $network !== null) {
        $network = strtoupper($network);
        if (isset($networkMap[$currency][$network])) {
            return $networkMap[$currency][$network];
        }
    }
    
    // Return original currency if no network mapping exists
    return $currency;
}

// Generate unique deposit ID
function generateDepositId() {
    return 'DEP_' . strtoupper(substr(md5(uniqid(rand(), true)), 0, 15));
}

// Make NOWPayments API request
function makeNowPaymentsRequest($endpoint, $data = null, $method = 'GET') {
    global $nowpayments_api_key, $nowpayments_base_url;
    
    $url = $nowpayments_base_url . '/' . ltrim($endpoint, '/');
    
    $headers = [
        'x-api-key: ' . $nowpayments_api_key,
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    if ($method === 'POST' && $data) {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        error_log("CURL Error: $curlError");
        return false;
    }
    
    if ($httpCode !== 200 && $httpCode !== 201) {
        error_log("NOWPayments API Error: HTTP $httpCode - Response: $response");
        $errorData = json_decode($response, true);
        if ($errorData) {
            return $errorData;
        }
        return false;
    }
    
    return json_decode($response, true);
}

// Get available currencies from NOWPayments
function getAvailableCurrencies() {
    $result = makeNowPaymentsRequest('currencies');
    if ($result && isset($result['currencies'])) {
        return $result['currencies'];
    }
    return [];
}

// Create payment with NOWPayments
function createPayment($amount, $currency_from, $currency_to, $order_id) {
    $domain = $_SERVER['HTTP_HOST'];
    
    $data = [
        'price_amount' => (float)$amount,
        'price_currency' => strtolower($currency_from),
        'pay_currency' => strtolower($currency_to),
        'order_id' => $order_id,
        'order_description' => 'Deposit to account',
        'ipn_callback_url' => "https://$domain/api/helper2/api_nowipn.php",
        'success_url' => "https://$domain/api/helper2/api_deposit.php?success=1",
        'cancel_url' => "https://$domain/api/helper2/api_deposit.php?canceled=1",
        'is_fee_paid_by_user' => true
    ];
    
    error_log("NOWPayments Request: " . json_encode($data));
    $result = makeNowPaymentsRequest('payment', $data, 'POST');
    error_log("NOWPayments Response: " . json_encode($result));
    return $result;
}

try {
    $deposit_id = generateDepositId();
    
    // Convert currency + network to proper NOWPayments currency code
    $payCurrency = getCurrencyCode($currency, $network);
    error_log("Converted currency '$currency' with network '$network' to: $payCurrency");
    
    // Optional: Verify the currency is supported by NOWPayments
    $availableCurrencies = getAvailableCurrencies();
    if (!empty($availableCurrencies) && !in_array($payCurrency, $availableCurrencies)) {
        error_log("Currency '$payCurrency' not found in available currencies. Available: " . implode(', ', $availableCurrencies));
        echo json_encode([
            'success' => false,
            'error' => "Currency $payCurrency is not supported. Please try a different network."
        ]);
        exit();
    }
    
    // Create NOWPayments payment
    $payment = createPayment($amount, 'USD', $payCurrency, $deposit_id);
    
    if ($payment && isset($payment['payment_id']) && isset($payment['pay_address'])) {
        // Double-check discord_id before insert
        if (empty($discord_id)) {
            throw new Exception("Discord ID is empty before database insert");
        }
        
        error_log("Inserting deposit - Address: {$payment['pay_address']}, Amount: " . ($payment['pay_amount'] ?? $amount) . ", Owner: $discord_id");
        
        // Insert deposit record into database
        $stmt = $pdo->prepare("
            INSERT INTO deposit (created_at, address, depo_status, amount, coin, depo_owner) 
            VALUES (NOW(), ?, 'pending', ?, ?, ?)
        ");
        $stmt->execute([
            $payment['pay_address'],
            $payment['pay_amount'] ?? $amount,
            strtoupper($currency),  // Store original currency (USDT, USDC, etc.)
            $discord_id
        ]);
        
        $deposit_db_id = $pdo->lastInsertId();
        
        // Return success with payment details
        echo json_encode([
            'success' => true,
            'payment' => [
                'payment_id' => $payment['payment_id'],
                'pay_address' => $payment['pay_address'],
                'pay_amount' => $payment['pay_amount'],
                'pay_currency' => strtoupper($payment['pay_currency']),
                'order_id' => $deposit_id,
                'deposit_db_id' => $deposit_db_id,
                'network' => $network  // Include network for frontend reference
            ]
        ]);
    } else {
        // Handle API error
        $errorMsg = 'Failed to create payment';
        
        if ($payment && isset($payment['message'])) {
            $errorMsg = $payment['message'];
        } elseif ($payment && isset($payment['error'])) {
            $errorMsg = $payment['error'];
        }
        
        echo json_encode([
            'success' => false,
            'error' => $errorMsg
        ]);
    }
    
} catch (Exception $e) {
    error_log("Deposit processing error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => 'Database error occurred'
    ]);
}
?>