<?php
require __DIR__ . '/../../db/config.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user'])) {
    echo json_encode([
        'success' => false,
        'message' => 'You must be logged in to make a purchase'
    ]);
    exit;
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request data'
    ]);
    exit;
}

$gamepassId = $data['gamepass_id'] ?? null;
$robuxAmount = $data['robux_amount'] ?? null;
$payAmount = $data['pay_amount'] ?? null;
$gamepassPrice = $data['gamepass_price'] ?? null;

// Validate input
if (!$gamepassId || !$robuxAmount || !$payAmount || !$gamepassPrice) {
    echo json_encode([
        'success' => false,
        'message' => 'Missing required fields'
    ]);
    exit;
}

// Validate amounts against config
if ($robuxAmount < $minrbx || $robuxAmount > $maxrbx) {
    echo json_encode([
        'success' => false,
        'message' => "Robux amount must be between {$minrbx} and {$maxrbx}"
    ]);
    exit;
}

if ($payAmount < $minpay || $payAmount > $maxpay) {
    echo json_encode([
        'success' => false,
        'message' => "Payment amount must be between \${$minpay} and \${$maxpay}"
    ]);
    exit;
}

// Verify the calculation is correct
$expectedRobux = floor(($payAmount / $rbxrate) * 1000);
if (abs($expectedRobux - $robuxAmount) > 10) { // Allow small variance
    echo json_encode([
        'success' => false,
        'message' => 'Invalid robux calculation'
    ]);
    exit;
}

$expectedGamepassPrice = round($robuxAmount / 0.7);
if (abs($expectedGamepassPrice - $gamepassPrice) > 1) { // Allow 1 robux variance
    echo json_encode([
        'success' => false,
        'message' => 'Invalid gamepass price calculation'
    ]);
    exit;
}

$discordId = $_SESSION['user']['discord_id'];

try {
    // Start transaction
    $pdo->beginTransaction();
    
    // Check user balance AND ban status
    $stmt = $pdo->prepare("SELECT balance, banned, reason FROM users WHERE discord_id = ? FOR UPDATE");
    $stmt->execute([$discordId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($user === false) {
        throw new Exception('User not found');
    }
    
    // Check if user is banned
    if ($user['banned'] === 'true') {
        $banReason = $user['reason'] ? ' Reason: ' . $user['reason'] : '';
        throw new Exception('You are banned from making purchases.' . $banReason);
    }
    
    $balance = $user['balance'];
    
    if ($balance < $payAmount) {
        throw new Exception('Insufficient balance');
    }
    
    // Verify gamepass one more time from Roblox API
    $gamepassUrl = "https://apis.roblox.com/game-passes/v1/game-passes/{$gamepassId}/product-info";
    $ch = curl_init($gamepassUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception('Failed to verify gamepass');
    }
    
    $gamepassData = json_decode($response, true);
    
    if (!$gamepassData || !isset($gamepassData['PriceInRobux'])) {
        throw new Exception('Invalid gamepass data');
    }
    
    if ($gamepassData['PriceInRobux'] != $gamepassPrice) {
        throw new Exception('Gamepass price mismatch');
    }
    
    if (!$gamepassData['IsForSale']) {
        throw new Exception('Gamepass is not for sale');
    }
    
    // Deduct balance
    $newBalance = $balance - $payAmount;
    $stmt = $pdo->prepare("UPDATE users SET balance = ? WHERE discord_id = ?");
    $stmt->execute([$newBalance, $discordId]);
    
    // Create order
    $stmt = $pdo->prepare("
        INSERT INTO orders (puser, order_status, gamepass_id, paid_amt) 
        VALUES (?, 'pending', ?, ?)
    ");
    $stmt->execute([$discordId, $gamepassId, $payAmount]);
    
    $orderId = $pdo->lastInsertId();
    
    // Commit transaction
    $pdo->commit();
    
    // Return success
    echo json_encode([
        'success' => true,
        'order_id' => $orderId,
        'message' => 'Order created successfully'
    ]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>